-- Copyright 2024-2025 by Todd Hundersmarck (ThundR) 
-- All Rights Reserved

--[[

Unauthorized use and/or distribution of this work entitles
myself, the author, to unlimited free and unrestricted use,
access, and distribution of any works related to the unauthorized
user and/or distributor.

--]]

local thModName = g_thRowCropSystem.coreData.mod.name
THVSpec_RowCropWorkArea = {}
THVSpec_RowCropWorkArea.CLASS_NAME = "THVSpec_RowCropWorkArea"
THVSpec_RowCropWorkArea.SPEC_NAME = "thRowCropWorkArea"
THVSpec_RowCropWorkArea.SPEC_KEY = string.format("spec_%s.%s", thModName, THVSpec_RowCropWorkArea.SPEC_NAME)
THVSpec_RowCropWorkArea.XML_BASE_KEY = "vehicle." .. THVSpec_RowCropWorkArea.SPEC_NAME
THVSpec_RowCropWorkArea.DATA_KEY = tostring(THVSpec_RowCropWorkArea)
THVSpec_RowCropWorkArea.CONFIG_NAME = {
    WORKAREA = "thWorkArea",
    ROW_SHUTOFF = "thRowShutoff"
}
THVSpec_RowCropWorkArea.CONFIG_BASE_KEY = {}
THVSpec_RowCropWorkArea.CONFIG_KEY = {}
for configId, configName in pairs(THVSpec_RowCropWorkArea.CONFIG_NAME) do
    THVSpec_RowCropWorkArea.CONFIG_BASE_KEY[configId] = string.format("%s.%sConfigurations", THVSpec_RowCropWorkArea.XML_BASE_KEY, configName)
    THVSpec_RowCropWorkArea.CONFIG_KEY[configId] = string.format("%s.%sConfiguration", THVSpec_RowCropWorkArea.CONFIG_BASE_KEY[configId], configName)
end
THVSpec_RowCropWorkArea.BLINKING_ERROR = {
    OLD_RCR_WORK_AREA = 1,
    MIS_ALIGNED_WORK_AREA = 2,
    MIS_ALIGNED_WORK_AREA_Y = 3,
    REVERSED_WORK_AREA = 4,
    ROTATED_WORK_AREA = 5
}
THUtils.createEnumTable(THVSpec_RowCropWorkArea.BLINKING_ERROR)
local debugFlagId = THUtils.createDebugFlagId(THVSpec_RowCropWorkArea.CLASS_NAME)
THVSpec_RowCropWorkArea.debugFlagId = debugFlagId
local function initScript()
    THUtils.overwriteFunction("ConfigurationUtil", "getConfigurationsFromXML", false, false, nil, THVSpec_RowCropWorkArea.gOverwrite_getConfigurationsFromXML)
end
local function getSpecTable(self)
    return THUtils.call(function()
        if self ~= nil then
            return self[THVSpec_RowCropWorkArea.SPEC_KEY]
        end
    end)
end
THVSpec_RowCropWorkArea.getSpecTable = getSpecTable
function THVSpec_RowCropWorkArea.initSpecialization()
    local specName = THVSpec_RowCropWorkArea.SPEC_NAME
    local xmlSchema = Vehicle.xmlSchema
    g_vehicleConfigurationManager:addConfigurationType(THVSpec_RowCropWorkArea.CONFIG_NAME.WORKAREA, g_i18n:getText("thConfiguration_rowWorkArea"), specName, VehicleConfigurationItem)
    g_vehicleConfigurationManager:addConfigurationType(THVSpec_RowCropWorkArea.CONFIG_NAME.ROW_SHUTOFF, g_i18n:getText("thConfiguration_rowShutoff"), specName, VehicleConfigurationItem)
    local xmlBasePath = THVSpec_RowCropWorkArea.XML_BASE_KEY
    local workAreaConfigPath = THVSpec_RowCropWorkArea.CONFIG_KEY.WORKAREA .. "(?)"
    local function registerWorkAreaPaths(pBasePath)
        local workAreaPath = pBasePath .. ".workArea(?)"
        xmlSchema:register(XMLValueType.STRING, workAreaPath .. "#indices", "Work area indices that use this configuration")
        local rowSpacingPath = workAreaPath .. ".rowSpacing(?)"
        xmlSchema:register(XMLValueType.FLOAT, rowSpacingPath .. "#width", "Space between rows")
        xmlSchema:register(XMLValueType.FLOAT, rowSpacingPath .. "#length", "Space between plants within row")
        xmlSchema:register(XMLValueType.INT, rowSpacingPath .. "#numRows", "Total number of rows")
        xmlSchema:register(XMLValueType.INT, rowSpacingPath .. "#skipRowOffset", "Number of rows to paint before skipping", 0)
        xmlSchema:register(XMLValueType.INT, rowSpacingPath .. "#numSkipRows", "Number of rows to skip", 0)
        xmlSchema:register(XMLValueType.BOOL, rowSpacingPath .. "#skipRowsFirst", "Start with skipped rows", false)
        xmlSchema:register(XMLValueType.BOOL, rowSpacingPath .. "#betweenRows", "Paint between rows", false)
        xmlSchema:register(XMLValueType.STRING, rowSpacingPath .. "#fruitTypes", "Fruit type names allowed to use this spacing")
        xmlSchema:register(XMLValueType.STRING, rowSpacingPath .. "#fruitTypeCategories", "Fruit type categories allowed to use this spacing")
    end
    registerWorkAreaPaths(xmlBasePath)
    registerWorkAreaPaths(workAreaConfigPath)
end
function THVSpec_RowCropWorkArea.prerequisitesPresent(specializations)
    if SpecializationUtil.hasSpecialization(WorkArea, specializations) then
        return true
    end
    return false
end
function THVSpec_RowCropWorkArea.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", THVSpec_RowCropWorkArea)
    SpecializationUtil.registerEventListener(vehicleType, "onRegisterActionEvents", THVSpec_RowCropWorkArea)
    SpecializationUtil.registerEventListener(vehicleType, "onUpdateTick", THVSpec_RowCropWorkArea)
    SpecializationUtil.registerEventListener(vehicleType, "onStartWorkAreaProcessing", THVSpec_RowCropWorkArea)
end
function THVSpec_RowCropWorkArea.registerOverwrittenFunctions(vehicleType)
    if SpecializationUtil.hasSpecialization(SowingMachine, vehicleType.specializations) then
        SpecializationUtil.registerOverwrittenFunction(vehicleType, "processSowingMachineArea", THVSpec_RowCropWorkArea.overwrite_processSowingMachineArea)
    end
end
function THVSpec_RowCropWorkArea.onLoad(self, savegame)
    THUtils.pcall(function()
        local specTable = getSpecTable(self)
        local workAreaSpec = self.spec_workArea
        specTable.rowCropWorkAreas = {}
        specTable.rowCropWorkAreasArray = {}
        if specTable.actionEvents == nil then
            specTable.actionEvents = {}
        end
        local rowShutoffConfigName = THVSpec_RowCropWorkArea.CONFIG_NAME.ROW_SHUTOFF
        local rowShutoffConfigIndex = THUtils.getNoNil(self.configurations[rowShutoffConfigName], 1)
        specTable.hasRowShutoff = rowShutoffConfigIndex == 2
        specTable.isRowShutoffActive = specTable.hasRowShutoff == true
        specTable.lastIsRowShutoffActive = specTable.isRowShutoffActive
        specTable.numWorkAreasByType = {}
        specTable.activeBlinkingErrors = {}
        specTable.blinkingErrorText = {}
        local blinkingErrorsArray = THVSpec_RowCropWorkArea.BLINKING_ERROR:getValues()
        local invalidWorkAreaText = g_i18n:getText("thMessage_invalidWorkArea")
        for errorId, errorIndex in pairs(blinkingErrorsArray) do
            local errorText = g_i18n:getText("thMessage_" .. errorId)
            if errorIndex >= 1 and errorIndex <= 4 then
                errorText = string.format(invalidWorkAreaText, errorText)
            end
            specTable.blinkingErrorText[errorIndex] = errorText
        end
        if workAreaSpec ~= nil then
            local workAreasArray = workAreaSpec.workAreas
            if workAreasArray ~= nil then
                local function addTypedWorkArea(pWorkAreaType)
                    if pWorkAreaType ~= nil then
                        if specTable.numWorkAreasByType[pWorkAreaType] == nil then
                            specTable.numWorkAreasByType[pWorkAreaType] = 1
                        else
                            specTable.numWorkAreasByType[pWorkAreaType] = specTable.numWorkAreasByType[pWorkAreaType] + 1
                        end
                    end
                end
                for _, workAreaDesc in pairs(workAreasArray) do
                    if workAreaDesc.index ~= nil and workAreaDesc.index > 0 then
                        if workAreaDesc.type == WorkAreaType.SOWINGMACHINE then
                            local thWorkAreaData = { desc = workAreaDesc, rowSpacingValues = {}, currentSpacingValues = {}, isAdjusted = false }
                            specTable.rowCropWorkAreas[workAreaDesc.index] = thWorkAreaData
                            table.insert(specTable.rowCropWorkAreasArray, thWorkAreaData)
                            addTypedWorkArea(workAreaDesc.type)
                        end
                    end
                end
            end
        end
        local fruitMetersPerPixel = g_thMapTypeManager:getFruitMetersPerPixel()
        local wrappedXML = THUtils.wrapXMLFile(self.xmlFile)
        if workAreaSpec ~= nil and wrappedXML ~= nil and fruitMetersPerPixel > 0 then
            local xmlBaseKey = THVSpec_RowCropWorkArea.XML_BASE_KEY
            local workAreaConfigName = THVSpec_RowCropWorkArea.CONFIG_NAME.WORKAREA
            local workAreaConfigIndex = THUtils.getNoNil(self.configurations[workAreaConfigName], 1)
            local workAreaConfigKey = string.format("%s(%d)", THVSpec_RowCropWorkArea.CONFIG_KEY.WORKAREA, workAreaConfigIndex - 1)
            if not wrappedXML:hasProperty(workAreaConfigKey) then
                workAreaConfigKey = xmlBaseKey
            end
            wrappedXML:iterate(workAreaConfigKey .. ".workArea", function(_, pWorkAreaKey)
                local workAreaIndices = wrappedXML:getValue(pWorkAreaKey .. "#indices")
                if workAreaIndices ~= nil and workAreaIndices ~= "" then
                    local workAreasArray = THUtils.splitString(workAreaIndices, " ")
                    local validWorkAreas = {}
                    if workAreasArray ~= nil and #workAreasArray > 0 then
                        for _, areaIndexString in ipairs(workAreasArray) do
                            local areaIndex = THUtils.toNumber(areaIndexString, true)
                            if areaIndex == nil then
                                THUtils.objectXMLErrorMsg(self, pWorkAreaKey, nil, THMessage.INVALID_VALUE, "work area index", areaIndex)
                            else
                                local thWorkAreaData = specTable.rowCropWorkAreas[areaIndex]
                                if thWorkAreaData == nil then
                                    THUtils.objectXMLErrorMsg(self, pWorkAreaKey, false, "Work area [%d] is not supported by this system", areaIndex)
                                else
                                    validWorkAreas[areaIndex] = thWorkAreaData
                                end
                            end
                        end
                    end
                    if next(validWorkAreas) ~= nil then
                        wrappedXML:iterate(pWorkAreaKey .. ".rowSpacing", function(_, pSpacingKey)
                            local spacingWidth = wrappedXML:getValue(pSpacingKey .. "#width")
                            local numRows = wrappedXML:getValue(pSpacingKey .. "#numRows")
                            local skipRowOffset = wrappedXML:getValue(pSpacingKey .. "#skipRowOffset")
                            local numSkipRows = wrappedXML:getValue(pSpacingKey .. "#numSkipRows")
                            local skipRowsFirst = wrappedXML:getValue(pSpacingKey .. "#skipRowsFirst", false)
                            local isBetweenRows = wrappedXML:getValue(pSpacingKey .. "#betweenRows", false)
                            local fruitTypeNames = wrappedXML:getValue(pSpacingKey .. "#fruitTypes")
                            local fruitTypeCategoryNames = wrappedXML:getValue(pSpacingKey .. "#fruitTypeCategories")
                            local allowedFruitTypes = nil
                            local isEntryValid = true
                            if isEntryValid and spacingWidth ~= nil then
                                if spacingWidth <= 0 then
                                    THUtils.objectXMLErrorMsg(self, pSpacingKey, nil, THMessage.VALUE_GREATER, "width", spacingWidth, 0)
                                    isEntryValid = false
                                end
                            end
                            if isEntryValid and numRows ~= nil then
                                if numRows <= 0 then
                                    THUtils.objectXMLErrorMsg(self, pSpacingKey, nil, THMessage.VALUE_GREATER, "numRows", numRows, 0)
                                    isEntryValid = false
                                end
                            end
                            if isEntryValid and (skipRowOffset ~= nil or numSkipRows ~= nil) then
                                if skipRowOffset ~= nil and skipRowOffset < 0 then
                                    THUtils.objectXMLErrorMsg(self, pSpacingKey, nil, THMessage.VALUE_GREATER_EQUAL, "skipRowOffset", skipRowOffset, 0)
                                    isEntryValid = false
                                elseif numSkipRows ~= nil and numSkipRows < 0 then
                                    THUtils.objectXMLErrorMsg(self, pSpacingKey, nil, THMessage.VALUE_GREATER_EQUAL, "numSkipRows", numSkipRows, 0)
                                    isEntryValid = false
                                else
                                    numSkipRows = numSkipRows or 0
                                    if numSkipRows == 0 then
                                        skipRowOffset = 0
                                        skipRowsFirst = false
                                    else
                                        skipRowOffset = THUtils.floor(skipRowOffset or 0)
                                        skipRowOffset = math.max(1, skipRowOffset)
                                    end
                                end
                            end
                            if isEntryValid and fruitTypeCategoryNames ~= nil and fruitTypeCategoryNames ~= "" then
                                local fruitTypeList = g_fruitTypeManager:getFruitTypeIndicesByCategoryNames(fruitTypeCategoryNames)
                                if fruitTypeList ~= nil then
                                    allowedFruitTypes = allowedFruitTypes or {}
                                    for _, fruitTypeIndex in pairs(fruitTypeList) do
                                        local rcsFruitData = g_thRowCropSystem:getRowCropFruitType(fruitTypeIndex)
                                        if rcsFruitData ~= nil then
                                            allowedFruitTypes[rcsFruitData.fruitType.index] = true
                                        end
                                    end
                                end
                            end
                            if isEntryValid and fruitTypeNames ~= nil and fruitTypeNames ~= "" then
                                local fruitTypeList = g_fruitTypeManager:getFruitTypeIndicesByNames(fruitTypeNames)
                                if fruitTypeList ~= nil then
                                    allowedFruitTypes = allowedFruitTypes or {}
                                    for _, fruitTypeIndex in pairs(fruitTypeList) do
                                        local rcsFruitData = g_thRowCropSystem:getRowCropFruitType(fruitTypeIndex)
                                        if rcsFruitData ~= nil then
                                            allowedFruitTypes[rcsFruitData.fruitType.index] = true
                                        end
                                    end
                                end
                            end
                            if isEntryValid then
                                for _, thWorkAreaData in pairs(validWorkAreas) do
                                    local areaSpacingValues = {
                                        numRows = numRows,
                                        spacingWidth = spacingWidth,
                                        skipRowOffset = skipRowOffset,
                                        numSkipRows = numSkipRows,
                                        skipRowsFirst = skipRowsFirst == true,
                                        isBetweenRows = isBetweenRows == true,
                                        useRowShutoff = specTable.hasRowShutoff == true,
                                        allowedFruitTypes = {}
                                    }
                                    if allowedFruitTypes ~= nil then
                                        for fruitTypeIndex in pairs(allowedFruitTypes) do
                                            areaSpacingValues.allowedFruitTypes[fruitTypeIndex] = true
                                        end
                                    end
                                    table.insert(thWorkAreaData.rowSpacingValues, areaSpacingValues)
                                end
                            end
                        end)
                    end
                end
            end)
        end
        if THUtils.getIsDebugEnabled() then
            THUtils.objectDisplayMsg(self, "THVSpec_RowCropWorkArea information:")
            THUtils.printTableValues(specTable)
            if next(specTable.rowCropWorkAreasArray) ~= nil then
                THUtils.displayMsg("Row crop work areas:")
                for _, thWorkAreaData in ipairs(specTable.rowCropWorkAreasArray) do
                    THUtils.printTable(thWorkAreaData, 1)
                    if next(thWorkAreaData.rowSpacingValues) ~= nil then
                        THUtils.displayMsg("Row spacing values:")
                        THUtils.printTable(thWorkAreaData.rowSpacingValues, 2)
                    end
                end
            end
        end
    end)
end
function THVSpec_RowCropWorkArea.onRegisterActionEvents(self, ...)
    THUtils.pcall(function()
        local specTable = getSpecTable(self)
        if self.isClient then
            self:clearActionEventsTable(specTable.actionEvents)
            if specTable.hasRowShutoff then
                local _, eventId = self:addActionEvent(specTable.actionEvents, InputAction.THRCS_TOGGLE_ROW_SHUTOFF, self, THVSpec_RowCropWorkArea.actionToggleRowShutoff, false, true, false, true)
                g_inputBinding:setActionEventTextPriority(eventId, GS_PRIO_HIGH)
                THVSpec_RowCropWorkArea.updateActionEvents(self)
            end
        end
    end)
end
function THVSpec_RowCropWorkArea.onUpdateTick(self, dt, isActiveForInput, ...)
    THUtils.pcall(function()
        local mission = g_thRowCropSystem.coreData.mission
        local specTable = getSpecTable(self)
        local isActiveForInputIgnoreAI = self:getIsActiveForInput(true, true)
        for errIndex in pairs(specTable.activeBlinkingErrors) do
            if self.isClient and mission ~= nil and isActiveForInputIgnoreAI then
                local errorMsg = specTable.blinkingErrorText[errIndex]
                if errorMsg ~= nil then
                    mission:showBlinkingWarning(errorMsg, 5000)
                end
            end
            specTable.activeBlinkingErrors[errIndex] = nil
        end
        if self.isClient then
            if specTable.isRowShutoffActive ~= nil
                and specTable.isRowShutoffActive ~= specTable.lastIsRowShutoffActive
            then
                THVSpec_RowCropWorkArea.updateActionEvents(self)
                specTable.lastIsRowShutoffActive = specTable.isRowShutoffActive
            end
        end
    end)
end
function THVSpec_RowCropWorkArea.onStartWorkAreaProcessing(self, dt, workAreasArray, ...)
    THUtils.pcall(function()
        local specTable = getSpecTable(self)
        THUtils.clearTable(specTable.activeBlinkingErrors)
    end)
end
function THVSpec_RowCropWorkArea.actionToggleRowShutoff(self)
    THUtils.pcall(function()
        local specTable = getSpecTable(self)
        if self.isClient then
            local nextIsRowShutoffActive = specTable.hasRowShutoff and not specTable.isRowShutoffActive
            THToggleRowShutoffEvent.sendEvent(self, nextIsRowShutoffActive)
        end
    end)
end
function THVSpec_RowCropWorkArea.showBlinkingError(self, errorIndex)
    local specTable = getSpecTable(self)
    if specTable.blinkingErrorText[errorIndex] == nil then
        THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "errorIndex", errorIndex)
    else
        specTable.activeBlinkingErrors[errorIndex] = true
        self:raiseActive()
    end
end
function THVSpec_RowCropWorkArea.updateActionEvents(self)
    local specTable = getSpecTable(self)
    local isActiveForInput = self:getIsActiveForInput(false, true)
    for _, eventDesc in pairs(specTable.actionEvents) do
        local eventId = eventDesc.actionEventId
        if eventId ~= nil then
            local isEventActive = false
            if eventDesc == specTable.actionEvents[InputAction.THRCS_TOGGLE_ROW_SHUTOFF] then
                if self.isClient and isActiveForInput and specTable.hasRowShutoff then
                    local eventText = g_i18n:getText("thAction_enableRowShutoff")
                    if specTable.isRowShutoffActive then
                        eventText = g_i18n:getText("thAction_disableRowShutoff")
                    end
                    g_inputBinding:setActionEventText(eventDesc.actionEventId, eventText)
                    isEventActive = true
                end
            end
            g_inputBinding:setActionEventActive(eventId, isEventActive)
        end
    end
end
function THVSpec_RowCropWorkArea.setIsRowShutoffEnabled(self, isEnabled)
    local specTable = getSpecTable(self)
    if THUtils.argIsValid(not isEnabled or isEnabled == true, "isEnabled", isEnabled) then
        if specTable.hasRowShutoff then
            specTable.isRowShutoffActive = isEnabled == true
        else
            specTable.isRowShutoffActive = false
        end
        THVSpec_RowCropWorkArea.updateActionEvents(self)
    end
end
function THVSpec_RowCropWorkArea.getRowCropWorkArea(self, workAreaId)
    local specTable = getSpecTable(self)
    local idVarType = type(workAreaId)
    if specTable ~= nil then
        if idVarType == THValueType.TABLE and workAreaId.index ~= nil then
            return specTable.rowCropWorkAreas[workAreaId.index]
        elseif idVarType == THValueType.NUMBER then
            return specTable.rowCropWorkAreas[workAreaId]
        elseif workAreaId ~= nil then
            THUtils.errorMsg(true, THMessage.ARGUMENT_INVALID, "workAreaId", workAreaId)
        end
    end
end
function THVSpec_RowCropWorkArea.getWorkAreaRowSpacing(self, workAreaId, fruitTypeId)
    local specTable = getSpecTable(self)
    local workAreaSpec = self.spec_workArea
    if specTable ~= nil and workAreaSpec ~= nil then
        local thWorkAreaData = THVSpec_RowCropWorkArea.getRowCropWorkArea(self, workAreaId)
        local fruitSpacingValues = g_thRowCropSystem:getFruitTypeRowSpacing(fruitTypeId, true)
        if thWorkAreaData ~= nil and fruitSpacingValues ~= nil then
            local currentValues = THUtils.clearTable(thWorkAreaData.currentSpacingValues)
            local rcsFruitData = fruitSpacingValues.fruitType
            local numSpacingEntries = #thWorkAreaData.rowSpacingValues
            if numSpacingEntries > 0 then
                for spacingIndex = 1, numSpacingEntries do
                    local spacingEntry = thWorkAreaData.rowSpacingValues[spacingIndex]
                    local isEntryValid = true
                    if isEntryValid and next(spacingEntry.allowedFruitTypes) ~= nil then
                        if rcsFruitData == nil or not spacingEntry.allowedFruitTypes[rcsFruitData.fruitType.index] then
                            isEntryValid = false
                        end
                    end
                    if isEntryValid then
                        if spacingEntry.isBetweenRows then
                            if thWorkAreaData.desc.type == WorkAreaType.SOWINGMACHINE then
                                isEntryValid = false
                            end
                        end
                    end
                    if isEntryValid then
                        for key, value in pairs(spacingEntry) do
                            currentValues[key] = value
                        end
                    end
                end
            end
            currentValues.workArea = thWorkAreaData
            local hasRowShutoff = THUtils.getNoNil(currentValues.useRowShutoff, specTable.hasRowShutoff)
            if hasRowShutoff then
                currentValues.useRowShutoff = specTable.isRowShutoffActive == true
            else
                currentValues.useRowShutoff = false
            end
            for key, value in pairs(fruitSpacingValues) do
                if currentValues[key] == nil then
                    currentValues[key] = value
                end
            end
            return currentValues
        end
    end
end
function THVSpec_RowCropWorkArea.updateRowCropWorkArea(self, workArea, fruitTypeId)
    local specTable = getSpecTable(self)
    if THUtils.argIsValid(type(workArea) == THValueType.TABLE, "workArea", workArea) then
        local fruitMetersPerPixel = g_thMapTypeManager:getFruitMetersPerPixel()
        local thWorkAreaData = THVSpec_RowCropWorkArea.getRowCropWorkArea(self, workArea)
        local areaSpacingValues = THVSpec_RowCropWorkArea.getWorkAreaRowSpacing(self, workArea, fruitTypeId)
        if thWorkAreaData ~= nil and fruitMetersPerPixel > 0
            and type(workArea.start) == THValueType.NUMBER and workArea.start > 0 and type(workArea.width) == THValueType.NUMBER and workArea.width > 0 and type(workArea.height) == THValueType.NUMBER and workArea.height > 0
        then
            local isDebugUpdateEnabled = THUtils.getIsDebugEnabled(debugFlagId, THDebugLevel.UPDATE)
            local rcsFruitData = nil
            if areaSpacingValues ~= nil then
                rcsFruitData = areaSpacingValues.fruitType
            end
            local lastFruitTypeIndex = thWorkAreaData.lastFruitTypeIndex or FruitType.UNKNOWN
            local isFruitTypeChanged = false
            if rcsFruitData == nil and lastFruitTypeIndex ~= FruitType.UNKNOWN then
                thWorkAreaData.lastFruitTypeIndex = FruitType.UNKNOWN
            elseif rcsFruitData ~= nil and lastFruitTypeIndex ~= rcsFruitData.fruitType.index then
                thWorkAreaData.lastFruitTypeIndex = rcsFruitData.fruitType.index
            end
            if thWorkAreaData.lastFruitTypeIndex ~= lastFruitTypeIndex then
                lastFruitTypeIndex = thWorkAreaData.lastFruitTypeIndex
                isFruitTypeChanged = true
            end
            if thWorkAreaData.defaultNodes == nil then
                local sx, sy, sz = getTranslation(workArea.start)
                local wx, wy, wz = getTranslation(workArea.width)
                local hx, hy, hz = getTranslation(workArea.height)
                thWorkAreaData.defaultNodes = { start = workArea.start, width = workArea.width, height = workArea.height }
                thWorkAreaData.defaultTranslation = { start = { sx, sy, sz }, width = { wx, wy, wz }, height = { hx, hy, hz } }
                thWorkAreaData.isAdjusted = false
            elseif thWorkAreaData.isAdjusted
                and (areaSpacingValues == nil or isFruitTypeChanged)
            then
                workArea.start = thWorkAreaData.defaultNodes.start
                workArea.width = thWorkAreaData.defaultNodes.width
                workArea.height = thWorkAreaData.defaultNodes.height
                setTranslation(workArea.start, THUtils.unpack(thWorkAreaData.defaultTranslation.start))
                setTranslation(workArea.width, THUtils.unpack(thWorkAreaData.defaultTranslation.width))
                setTranslation(workArea.height, THUtils.unpack(thWorkAreaData.defaultTranslation.height))
                thWorkAreaData.isAdjusted = false
                if isDebugUpdateEnabled then
                    THUtils.objectDisplayMsg(self, "Restoring default work area")
                    THUtils.displayMsg("")
                end
            end
            if areaSpacingValues ~= nil and rcsFruitData ~= nil then
                if not thWorkAreaData.isAdjusted then
                    local wsx, wsy, wsz = getWorldTranslation(workArea.start)
                    local wwx, wwy, wwz = getWorldTranslation(workArea.width)
                    local whx, why, whz = getWorldTranslation(workArea.height)
                    local lsx, lsy, lsz = 0, 0, 0
                    local lwx, lwy, lwz = worldToLocal(workArea.start, wwx, wwy, wwz)
                    local lhx, lhy, lhz = worldToLocal(workArea.start, whx, why, whz)
                    local workWidth = THUtils.getVector3Distance(lsx, lsy, lsz, lwx, lwy, lwz)
                    local workLength = THUtils.getVector3Distance(lsx, lsy, lsz, lhx, lhy, lhz)
                    if isDebugUpdateEnabled then
                        THUtils.objectDisplayMsg(self, "Work area base properties:")
                        THUtils.displayMsg("- lsx, lsy, lsz: %0.6f, %0.6f, %0.6f", lsx, lsy, lsz)
                        THUtils.displayMsg("- lwx, lwy, lwz: %0.6f, %0.6f, %0.6f", lwx, lwy, lwz)
                        THUtils.displayMsg("- lhx, lhy, lhz: %0.6f, %0.6f, %0.6f", lhx, lhy, lhz)
                        THUtils.displayMsg("")
                        THUtils.displayMsg("- workWidth: %0.6f", workWidth)
                        THUtils.displayMsg("- workLength: %0.6f", workLength)
                        THUtils.displayMsg("")
                    end
                    local isWorkAreaValid = true
                    if workWidth < fruitMetersPerPixel and workLength < fruitMetersPerPixel then
                        if workArea.type ~= nil then
                            local numTypedWorkAreas = specTable.numWorkAreasByType[workArea.type] or 0
                            if numTypedWorkAreas > 3 then
                                isWorkAreaValid = false
                                THVSpec_RowCropWorkArea.showBlinkingError(self, THVSpec_RowCropWorkArea.BLINKING_ERROR.OLD_RCR_WORK_AREA)
                            end
                        end
                    end
                    if isWorkAreaValid then
                        if math.abs(lwy) >= 0.001 or math.abs(lhy) >= 0.001
                            or math.abs(lhy - lwy) >= 0.001
                        then
                            isWorkAreaValid = false
                            THVSpec_RowCropWorkArea.showBlinkingError(self, THVSpec_RowCropWorkArea.BLINKING_ERROR.MIS_ALIGNED_WORK_AREA_Y)
                        elseif lhz > 0 then
                            isWorkAreaValid = false
                            THVSpec_RowCropWorkArea.showBlinkingError(self, THVSpec_RowCropWorkArea.BLINKING_ERROR.REVERSED_WORK_AREA)
                        end
                    end
                    if isWorkAreaValid then
                        local absWX = math.abs(lwx)
                        local absWZ = math.abs(lwz)
                        local absHX = math.abs(lhx)
                        local absHZ = math.abs(lhz)
                        if absWZ > absWX and absHX > absHZ then
                            isWorkAreaValid = false
                            THVSpec_RowCropWorkArea.showBlinkingError(self, THVSpec_RowCropWorkArea.BLINKING_ERROR.ROTATED_WORK_AREA)
                        elseif absWZ >= 0.001 or absHX >= 0.001 then
                            isWorkAreaValid = false
                            THVSpec_RowCropWorkArea.showBlinkingError(self, THVSpec_RowCropWorkArea.BLINKING_ERROR.MIS_ALIGNED_WORK_AREA)
                        end
                    end
                    if isWorkAreaValid
                        and workWidth > 0 and workLength > 0
                    then
                        local spacingWidth = math.max(fruitMetersPerPixel, areaSpacingValues.spacingWidth or 0)
                        local spacingLength = math.max(fruitMetersPerPixel, areaSpacingValues.spacingLength or 0)
                        local numRows = areaSpacingValues.numRows or 0
                        local isWorkAreaAligned = false
                        local dwx = lwx - lsx
                        local dwy = lwy - lsy
                        local dwz = lwz - lsz
                        local dhx = lhx - lsx
                        local dhy = lhy - lsy
                        local dhz = lhz - lsz
                        local alignedWidth = workWidth
                        if numRows > 0 then
                            alignedWidth = numRows * spacingWidth
                        else
                            local skipRowOffset = areaSpacingValues.skipRowOffset or 0
                            local numSkipRows = areaSpacingValues.numSkipRows or 0
                            if skipRowOffset > 0 and numSkipRows > 0 then
                                local layoutWidth = (skipRowOffset + numSkipRows) * spacingWidth
                                if layoutWidth > workWidth then
                                    alignedWidth = layoutWidth
                                else
                                    alignedWidth = THUtils.round(workWidth / layoutWidth) * layoutWidth
                                end
                            end
                        end
                        alignedWidth = math.max(spacingWidth, alignedWidth)
                        if math.abs(alignedWidth - workWidth) >= 0.001 then
                            if isDebugUpdateEnabled then
                                THUtils.objectDisplayMsg(self, "Aligning work area width:")
                                THUtils.displayMsg("- width (normal): %0.6f", workWidth)
                                THUtils.displayMsg("- width (aligned): %0.6f", alignedWidth)
                                if numRows > 0 then
                                    THUtils.displayMsg("- numRows: %s", numRows)
                                end
                                THUtils.displayMsg("")
                            end
                            local mwx = (lsx + lwx) / 2
                            local mwy = (lsy + lwy) / 2
                            local mwz = (lsz + lwz) / 2
                            local widthFactor = alignedWidth / workWidth
                            dwx = dwx * widthFactor
                            dwy = dwy * widthFactor
                            dwz = dwz * widthFactor
                            lsx = mwx - (dwx * 0.5)
                            lsy = mwy - (dwy * 0.5)
                            lsz = mwz - (dwz * 0.5)
                            lwx = lsx + dwx
                            lwy = lsy + dwy
                            lwz = lsz + dwz
                            lhx = lsx + dhx
                            lhy = lsy + dhy
                            lhz = lsz + dhz
                            workWidth = THUtils.getVector3Distance(lsx, lsy, lsz, lwx, lwy, lwz)
                            isWorkAreaAligned = true
                            if isDebugUpdateEnabled then
                                THUtils.displayMsg("- lsx, lsy, lsz: %0.6f, %0.6f, %0.6f", lsx, lsy, lsz)
                                THUtils.displayMsg("- lwx, lwy, lwz: %0.6f, %0.6f, %0.6f", lwx, lwy, lwz)
                                THUtils.displayMsg("- lhx, lhy, lhz: %0.6f, %0.6f, %0.6f", lhx, lhy, lhz)
                                THUtils.displayMsg("")
                                THUtils.displayMsg("- width (final): %0.6f", workWidth)
                                THUtils.displayMsg("")
                            end
                        end
                        local alignedLength = math.max(spacingLength, workLength)
                        if math.abs(alignedLength - workLength) >= 0.001 then
                            if isDebugUpdateEnabled then
                                THUtils.objectDisplayMsg(self, "Aligning work area length:")
                                THUtils.displayMsg("- length (normal): %0.6f", workLength)
                                THUtils.displayMsg("- length (aligned): %0.6f", alignedLength)
                                THUtils.displayMsg("")
                            end
                            local mhx = (lsx + lhx) / 2
                            local mhy = (lsy + lhy) / 2
                            local mhz = (lsz + lhz) / 2
                            local lengthFactor = alignedLength / workLength
                            dhx = dhx * lengthFactor
                            dhy = dhy * lengthFactor
                            dhz = dhz * lengthFactor
                            lsx = mhx - (dhx * 0.5)
                            lsy = mhy - (dhy * 0.5)
                            lsz = mhz - (dhz * 0.5)
                            lwx = lsx + dwx
                            lwy = lsy + dwy
                            lwz = lsz + dwz
                            lhx = lsx + dhx
                            lhy = lsy + dhy
                            lhz = lsz + dhz
                            workLength = THUtils.getVector3Distance(lsx, lsy, lsz, lhx, lhy, lhz)
                            isWorkAreaAligned = true
                            if isDebugUpdateEnabled then
                                THUtils.displayMsg("- lsx, lsy, lsz: %0.6f, %0.6f, %0.6f", lsx, lsy, lsz)
                                THUtils.displayMsg("- lwx, lwy, lwz: %0.6f, %0.6f, %0.6f", lwx, lwy, lwz)
                                THUtils.displayMsg("- lhx, lhy, lhz: %0.6f, %0.6f, %0.6f", lhx, lhy, lhz)
                                THUtils.displayMsg("")
                                THUtils.displayMsg("- length (final): %0.6f", workLength)
                                THUtils.displayMsg("")
                            end
                        end
                        if isWorkAreaAligned then
                            wsx, wsy, wsz = localToWorld(workArea.start, lsx, lsy, lsz)
                            wwx, wwy, wwz = localToWorld(workArea.start, lwx, lwy, lwz)
                            whx, why, whz = localToWorld(workArea.start, lhx, lhy, lhz)
                            setWorldTranslation(workArea.start, wsx, wsy, wsz)
                            setWorldTranslation(workArea.width, wwx, wwy, wwz)
                            setWorldTranslation(workArea.height, whx, why, whz)
                        end
                    end
                    thWorkAreaData.isAdjusted = true
                end
                return true, areaSpacingValues
            end
        end
    end
    return false
end
function THVSpec_RowCropWorkArea.addRowShutoffConfigToXML(xmlFile)
    if THUtils.argIsValid(type(xmlFile) == THValueType.TABLE, "xmlFile", xmlFile) then
        local specName = THVSpec_RowCropWorkArea.SPEC_NAME
        local configName = THVSpec_RowCropWorkArea.CONFIG_NAME.ROW_SHUTOFF
        local configBaseKey = string.format("vehicle.%s.%sConfigurations", specName, configName)
        local configTitle = g_i18n:getText("thConfiguration_rowShutoff")
        if xmlFile ~= nil and not xmlFile:hasProperty(configBaseKey) then
            xmlFile:setString(configBaseKey .. "#title", configTitle)
            xmlFile:setBool(configBaseKey .. "#isYesNoOption", true)
            for configIndex = 1, 2 do
                local configKey = string.format("%s.%sConfiguration(%d)", configBaseKey, configName, configIndex - 1)
                if configIndex == 1 then
                    xmlFile:setString(configKey .. "#name", "$l10n_configuration_valueNo")
                    xmlFile:setInt(configKey .. "#price", 0)
                    xmlFile:setBool(configKey .. ".rowShutoff#isActive", false)
                else
                    xmlFile:setString(configKey .. "#name", "$l10n_configuration_valueYes")
                    xmlFile:setInt(configKey .. "#price", 500)
                    xmlFile:setBool(configKey .. ".rowShutoff#isActive", true)
                end
            end
        end
    end
end
function THVSpec_RowCropWorkArea.overwrite_processSowingMachineArea(self, superFunc, workArea, dt, ...)
    local specTable = getSpecTable(self)
    local sowingMachineSpec = self.spec_sowingMachine
    local seedFruitType = nil
    local prependSuccess = false
    if specTable ~= nil and sowingMachineSpec ~= nil and workArea ~= nil then
        THUtils.pcall(function()
            seedFruitType = sowingMachineSpec.workAreaParameters.seedsFruitType
            local _, areaSpacingValues = THVSpec_RowCropWorkArea.updateRowCropWorkArea(self, workArea, seedFruitType)
            if areaSpacingValues ~= nil then
                g_thRowCropSystem:setFruitTypeRowSpacing(seedFruitType, areaSpacingValues)
                prependSuccess = true
            end
        end)
    end
    local function appendFunc(...)
        if prependSuccess then
            THUtils.pcall(function()
                g_thRowCropSystem:resetFruitTypeRowSpacing(seedFruitType)
            end)
        end
        return ...
    end
    return appendFunc(superFunc(self, workArea, dt, ...))
end
function THVSpec_RowCropWorkArea.gOverwrite_getConfigurationsFromXML(superFunc, manager, xmlFile, configType, baseDirectory, customEnv, flag1, storeItem, ...)
    local xmlFileType = type(xmlFile)
    if configType == "vehicle" and xmlFileType == THValueType.TABLE or (xmlFileType == THValueType.NUMBER and xmlFile > 0) then
        THUtils.pcall(function(...)
            local specName = THVSpec_RowCropWorkArea.SPEC_NAME
            local rowShutoffConfigName = THVSpec_RowCropWorkArea.CONFIG_NAME.ROW_SHUTOFF
            local rowShutoffBaseKey = string.format("vehicle.%s.%sConfigurations", specName, rowShutoffConfigName)
            local wrappedXML = THUtils.wrapXMLFile(xmlFile)
            if wrappedXML ~= nil and not wrappedXML:hasProperty(rowShutoffBaseKey) then
                local vehicleType = wrappedXML:getString("vehicle#type")
                if vehicleType ~= nil and vehicleType ~= "" then
                    local typeDesc = g_vehicleTypeManager:getTypeByName(vehicleType, customEnv)
                    if typeDesc ~= nil then
                        if typeDesc.specializations ~= nil then
                            if SpecializationUtil.hasSpecialization(SowingMachine, typeDesc.specializations) then
                                local hasValidWorkAreas = false
                                local function validateWorkAreas(pBaseKey)
                                    if not hasValidWorkAreas then
                                        local workAreaXMLIndex = 0
                                        while true do
                                            local workAreaKey = string.format("%s.workArea(%d)", pBaseKey, workAreaXMLIndex)
                                            if not wrappedXML:hasProperty(workAreaKey) then
                                                break
                                            end
                                            local workAreaType = wrappedXML:getString(workAreaKey .. "#type")
                                            if workAreaType ~= nil and workAreaType ~= "" then
                                                workAreaType = string.upper(workAreaType)
                                                if workAreaType == "SOWINGMACHINE" then
                                                    hasValidWorkAreas = true
                                                    return
                                                end
                                            end
                                            workAreaXMLIndex = workAreaXMLIndex + 1
                                        end
                                    end
                                end
                                local workAreasBaseKey = "vehicle.workAreas.workAreaConfigurations.workAreaConfiguration"
                                if wrappedXML:hasProperty(workAreasBaseKey .. "(0)") then
                                    local configXMLIndex = 0
                                    while true do
                                        local configKey = string.format("%s(%d)", workAreasBaseKey, configXMLIndex)
                                        if hasValidWorkAreas or not wrappedXML:hasProperty(configKey) then
                                            break
                                        end
                                        validateWorkAreas(configKey)
                                        configXMLIndex = configXMLIndex + 1
                                    end
                                else
                                    workAreasBaseKey = "vehicle.workAreas"
                                    validateWorkAreas(workAreasBaseKey)
                                end
                                if hasValidWorkAreas then
                                    THVSpec_RowCropWorkArea.addRowShutoffConfigToXML(wrappedXML)
                                end
                            end
                        end
                    end
                end
            end
        end, ...)
    end
    return superFunc(manager, xmlFile, configType, baseDirectory, customEnv, flag1, storeItem, ...)
end
THUtils.pcall(initScript)